/**
 * \file: exchnd_interface.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * User space interface to the exception handler device.
 * The interface encapsulates the ioctls to the exception handler device into
 * functions to be used by the daemon.
 *
 * \component: exchndd
 *
 * \author: Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>

#include <sys/ioctl.h>

#include <linux/exchnd.h>

#include "exchnd_interface.h"
#include "exchnd_names.h"

/* Align with the driver version we are built against.
 * Change in API implies major version increase. Minor version shall
 * not affect the interface.
 */
#define EXCHND_REQUIRED_VERSION_MIN (EXCHND_VERSION & 0xFF00)
#define EXCHND_REQUIRED_VERSION_MAX ((EXCHND_VERSION & 0xFF00) + 0x99)

static unsigned long exchnd_get_version(int exh_fd)
{
    unsigned long value;
    int rc = ioctl(exh_fd, IOCTL_EXCHND_VERSION, &value);
    return (rc >= 0) ? value : (unsigned long)~0;
}

int exchnd_check_version(int exh_fd, char *dev_name)
{
    /* check the version of exchnd Linux kernel driver */
    unsigned long version = exchnd_get_version(exh_fd);
    int rc = 0;

    if ((version < EXCHND_REQUIRED_VERSION_MIN) ||
        (version > EXCHND_REQUIRED_VERSION_MAX)) {
        /* the driver version is insufficient */
        printf("%s: wrong version v%lu.%02lX (0x%lX), "
               "expecting v%d.%02X ... v%d.%02X\n",
               dev_name, version >> 8, version & 0xff, version,
               EXCHND_REQUIRED_VERSION_MIN >> 8,
               EXCHND_REQUIRED_VERSION_MIN & 0xff,
               EXCHND_REQUIRED_VERSION_MAX >> 8,
               EXCHND_REQUIRED_VERSION_MAX & 0xff);
        rc = -ENODEV;
    }

    return rc;
}

int exchnd_create_device(char *dev_name, int flags)
{
    int exh_fd = 0;
    exh_fd = open(dev_name, flags);

    if (exh_fd < 0) {
        printf("Unable open %s: %s\n", dev_name, strerror(errno));
        return exh_fd;
    }

    if (exchnd_check_version(exh_fd, dev_name)) {
        close(exh_fd);
        return -ENODEV;
    }

    return exh_fd;
}

int exchnd_recover(int exh_fd)
{
#ifdef IOCTL_EXCHND_RECOVERY
    int rc = ioctl(exh_fd, IOCTL_EXCHND_RECOVERY);

    if (rc < 0)
        printf("Unable to call recovery: %s\n", strerror(errno));

    return rc;
#else
    (void)exh_fd;
    return 0;
#endif
}

int exchnd_configure_module(int exh_fd, struct exchnd_conf_param conf)
{
#ifdef IOCTL_EXCHND_MODULES
    int rc = ioctl(exh_fd, IOCTL_EXCHND_MODULES, &conf);

    if (rc < 0)
        printf("Unable to modify module configuration: %s\n", strerror(errno));

    return rc;
#else
    (void)exh_fd;
    (void)conf;
    return 0;
#endif
}

/*
 * \func exchnd_print_configuration
 *
 * Prints oot the configuration of a module to the console
 * The function retrieves the configuration of the modules for a trigger
 * from the exception handler kernel module via an ioctl. The retrieved
 * list of the modules is printed to stdout
 *
 * \param exh_fd File descriptor of the exchnd device.
 * \param trig Trigger the configuration shall be printed for
 */
void exchnd_print_configuration(int exh_fd, struct exchnd_conf_param mod_list)
{
    unsigned int loop;

    if (exh_fd > 0) {
        int res = ioctl(exh_fd,
                        IOCTL_EXCHND_CONFIGURATION_GET,
                        &mod_list);

        if (res != 0) {
            printf("ioctl failed with error: %s(%d)\n",
                   strerror(errno),
                   errno);
            return;
        }

        switch (mod_list.conf_type) {
        case EXCHND_CONF_SIGHDL_MASK:
            printf("0x%X\n", mod_list.sig_handled_mask);
            break;
        default:
        {
            enum exchnd_modules *modules = mod_list.modules;
            enum exchnd_modules id = EHM_NONE;

            /* Klocwork Fix */
            /* To use the Short circuit evaluation of logical AND, reordering  *
            * the conditions in the below for() so that if the value of loop  *
            * is more than EHM_LAST_ELEMENT then the condition modules[loop]  *
            * is not evaluated.                                               */
            for (loop = 0; (loop < EHM_LAST_ELEMENT) &&
                 (modules[loop] != EHM_NONE);
                 loop++) {
                id = modules[loop];
                printf(" %s", module_name(id));
            }

            printf("\n");
        }
        }
    } else {
        printf("Invalid file descriptor handle !\n");
    }
}

/*
 * \func exchnd_configuration_set
 *
 * Sets the configured modules for triggers or signals
 * The function sends the modules configured via an ioctl to the exception
 * handler module. The element to set modules to is transferred as first element
 * of the list of the configured modules.
 *
 * \param exh_fd File descriptor of the exchnd device.
 * \param module_list List of modules, trigger or signal as first element.
 */
void exchnd_configuration_set(int exh_fd, struct exchnd_conf_param conf)
{
    if (exh_fd) {
        int res = ioctl(exh_fd,
                        IOCTL_EXCHND_CONFIGURATION_SET,
                        &conf);

        if (res != 0)
            printf("ioctl failed with error: %s(%d)\n",
                   strerror(errno),
                   errno);
    }
}

/*
 * \func display_list Displays configuration in a human readable format
 *
 * \param mod_list The list to display
 * \param conf Configuration id user want to display, 0 to display config list.
 * \param type Trigger or Signal list ?
 */
void display_list(struct exchnd_conf_param mod_list, unsigned int conf)
{
    unsigned int start = 1;
    unsigned int end = (!conf) ? EC_LAST_ELEMENT : start + 1;
    int last_element = ET_LAST_ELEMENT;
    int type = mod_list.conf_type;

    if (type == EXCHND_CONF_SIGNAL_SET) {
        last_element = SIGSYS + 1;
        end = (!conf) ? ESC_LAST_ELEMENT : start + 1;
    }

    for (; start < end; start++) {
        int i = 1;
        char selected[1] = "";
        unsigned int id = (!conf) ? start : conf;

        if (!conf && (start == mod_list.signal[0][0]))
            selected[0] = '*';

        if (type == EXCHND_CONF_TRIGGER_SET) {
            enum exchnd_conf idt = (enum exchnd_conf)id;
            printf("%s%s\n", conf_name(idt), selected);
        } else {
            enum exchnd_signal_conf ids = (enum exchnd_signal_conf)id;
            printf("%s%s\n", signal_conf_name(ids), selected);
        }

        if (!conf)
            continue;

        for (; i < last_element; i++) {
            int j = 0;

            if (type == EXCHND_CONF_TRIGGER_SET) {
                enum exchnd_triggers idt = (enum exchnd_triggers)i;
                printf("    %s\n", trigger_name(idt));
            } else {
                printf("    %s\n", signal_name(i));
            }

            for (; mod_list.signal[i][j] != EHM_NONE; j++) {
                enum exchnd_modules mod =
                    (enum exchnd_modules)mod_list.signal[i][j];
                printf("        %s\n", module_name(mod));
            }

            if (j == 0) {
                enum exchnd_modules mod =
                    (enum exchnd_modules)mod_list.signal[i][j];
                printf("        %s\n", module_name(mod));
            }
        }

        printf("\n");
    }
}

/*
 * \func exchnd_print_def_conf
 *
 * Prints out the default configuration for triggers or signals to the console
 * The function retrieves EHMs for a given configuration from the exception
 * handler kernel module via an ioctl. The retrieved list of the modules is
 * printed to stdout.
 *
 * \param exh_fd File descriptor of the exchnd device.
 * \param conf The configuration that shall be printed
 */
void exchnd_print_def_conf(int exh_fd, struct exchnd_conf_param mod_list)
{
    /* Save this information as we'll loose it. */
    unsigned int conf = mod_list.modules[0];
    int res = ioctl(exh_fd,
                    IOCTL_EXCHND_CONFIGURATION_GET,
                    &mod_list);

    if (res == 0)
        display_list(mod_list, conf);
    else
        printf("ioctl failed with error: %s(%d)\n",
               strerror(errno),
               errno);
}

/*
 * \func exchnd_def_conf_set
 *
 * Sets the configured modules for a trigger
 * The function sends the modules configured for a trigger via an ioctl to
 * the exception handler module. The trigger is transferred as first
 * element of the list of the configured modules
 *
 * \param exh_fd File descriptor of the exchnd device.
 * \param module_list List of modules, trigger as first element.
 */
void exchnd_def_conf_set(int exh_fd, struct exchnd_conf_param trigger_conf)
{
    int res = ioctl(exh_fd,
                    IOCTL_EXCHND_CONFIGURATION_SET,
                    &trigger_conf);

    if (res != 0)
        printf("ioctl failed with error: %s(%d)\n",
               strerror(errno),
               errno);
}

/*
 * \func exchnd_modify_filter
 *
 * Add/remove processes to/from the driver's white list
 *
 * \param exh_fd File descriptor of the exchnd device.
 * \param process tab field with process name, process type, operation type
 * \param Number of processes to add/remove
 */
void exchnd_modify_filter(int exh_fd, struct exchnd_conf_filter conf)
{
    int res = 0;
    res = ioctl(exh_fd,
                IOCTL_EXCHND_MODIFY_FILTER,
                &conf);

    if (res != 0)
        printf("ioctl failed with error: %s(%d) on %s\n",
               strerror(errno),
               errno,
               conf.id);
}

/*
 * \func exchnd_config_app_specific
 *
 * Configure an application specific module with a specific library
 *
 * \param exh_fd File descriptor of the exchnd device.
 * \param app_spec_module The module to configure
 * \param library_path The path to the library
 *
 */
void exchnd_config_app_specific(int exh_fd,
                                enum exchnd_modules mod,
                                char *lib_path)
{
#if (EXCHND_VERSION > 0x199)
    int res = 0;
    struct exchnd_conf_app_spec conf;

    conf.module = mod;
    strncpy(conf.lib_path, lib_path, EXH_PATH_MAX);

    res = ioctl(exh_fd,
                IOCTL_EXCHND_APP_SPECIFIC,
                &conf);

    if (res != 0)
        printf("ioctl failed with error: %s(%d)\n",
               strerror(errno),
               errno);

#else
    (void)exh_fd;
    (void)mod;
    (void)lib_path;
    printf("Application specific module not available.");
#endif
}

/*
 * \func exchnd_print_triggers
 *
 * Prints out the available triggers to the console
 * The function prints out the names of the available trigger to stdout.
 * The names are the same that are used in the configuration of the trigger.
 */
void exchnd_print_triggers(void)
{
    enum exchnd_triggers loop;

    for (loop = ET_NONE; loop < ET_LAST_ELEMENT; loop++)
        printf("%s\n", trigger_name(loop));

    printf("\n");
}

/*
 * \func exchnd_print_modules
 *
 * Prints out the available modules to the console
 * The function prints out the names of the available modules to stdout.
 * The names are the same that are used in the configuration of the trigger.
 */
void exchnd_print_modules(void)
{
    enum exchnd_modules loop;

    for (loop = EHM_NONE; loop < EHM_LAST_ELEMENT; loop++)
        printf("%s\n", module_name(loop));

    printf("\n");
}

/*
 * \func exchnd_print_signals
 *
 * Prints out the available signals to the console
 * The function prints out the names of the available signals to stdout.
 * The names are the same that are used in the configuration of the trigger.
 */
void exchnd_print_signals(void)
{
    unsigned int loop;

    for (loop = 1; loop < SIGSYS + 1; loop++)
        printf("%s\n", signal_name(loop));

    printf("\n");
}

/*
 * \func exchnd_normalize
 *
 * Helper that normalize names received from user to have uppercase only ones.
 */
void exchnd_normalize(char *to_norm)
{
    int i = 0;

    for (i = 0; to_norm[i]; i++)
        if (isalpha(to_norm[i]))
            to_norm[i] = toupper(to_norm[i]);

}